/*:
 * @target MZ
 * @plugindesc Overrides tile passability by coordinates and optionally blocks region IDs if a switch is ON. DotMove compatible. | BitQuest Studio
 * @author BitQuest Studio
 *
 * @param RegionBlockSwitchId
 * @type switch
 * @desc Switch ID that enables region-based impassability.
 *
 * @param BlockedRegionIds
 * @type number[]
 * @desc List of region IDs that should become impassable when the switch is ON.
 *
 * @param EnableDebug
 * @text Enable Debug Logs
 * @desc Outputs debug logs to the developer console for testing
 * @type boolean
 * @default false
 */

(() => {
  const parameters = PluginManager.parameters("TileImpassableOverride");
  const RegionBlockSwitchId = Number(parameters.RegionBlockSwitchId || 0);
  const BlockedRegionIds = JSON.parse(parameters.BlockedRegionIds || "[]").map(Number);
  const DebugMode = parameters.EnableDebug === 'true';
  function log(...args) { if (DebugMode) console.log("[TileImpassableOverride]", ...args); }

  // Initialize plugin
  log(`Plugin loaded - Switch ID: ${RegionBlockSwitchId}, Blocked Regions: ${BlockedRegionIds.join(', ')}`);

  const _Game_Map_checkPassage = Game_Map.prototype.checkPassage;
  Game_Map.prototype.checkPassage = function(x, y, bit) {
    const key = `${x},${y}`;
    if (this._impassableOverrides?.[key]) return false;

    const blockingSwitch = RegionBlockSwitchId && $gameSwitches.value(RegionBlockSwitchId);
    const regionId = this.regionId(x, y);

    // Check if this region should be blocked
    if (blockingSwitch && BlockedRegionIds.includes(regionId)) {
      log(`Blocking region ${regionId} at (${x}, ${y}) - Switch ${RegionBlockSwitchId} is ON`);
      return false; // Block ALL movement through this region when switch is ON
    }

    return _Game_Map_checkPassage.call(this, x, y, bit);
  };

  // Override Game_Player's canPass to add additional blocking
  const _Game_Player_canPass = Game_Player.prototype.canPass;
  Game_Player.prototype.canPass = function(x, y, d) {
    const blockingSwitch = RegionBlockSwitchId && $gameSwitches.value(RegionBlockSwitchId);
    const regionId = $gameMap.regionId(x, y);

    if (blockingSwitch && BlockedRegionIds.includes(regionId)) {
      log(`Player blocked at region ${regionId} (${x}, ${y})`);
      return false;
    }

    return _Game_Player_canPass.call(this, x, y, d);
  };

  // DotMove compatibility
  if (window.DotMoveUtils) {
    const _DotMoveUtils_isCollidedTile = DotMoveUtils.isCollidedTile;
    DotMoveUtils.isCollidedTile = function(x, y, d, character) {
      const key = `${Math.floor(x)},${Math.floor(y)}`;
      if ($gameMap._impassableOverrides?.[key]) return true;

      const blockingSwitch = RegionBlockSwitchId && $gameSwitches.value(RegionBlockSwitchId);
      const regionId = $gameMap.regionId(Math.floor(x), Math.floor(y));

      if (blockingSwitch && BlockedRegionIds.includes(regionId)) {
        log(`DotMove: Blocking region ${regionId} at (${Math.floor(x)}, ${Math.floor(y)})`);
        return true; // Block movement through this region
      }

      return _DotMoveUtils_isCollidedTile.call(this, x, y, d, character);
    };
  }

  // Add a function to manually check if regions are being blocked
  window.TileImpassableOverride = {
    isRegionBlocked: function(x, y) {
      const blockingSwitch = RegionBlockSwitchId && $gameSwitches.value(RegionBlockSwitchId);
      const regionId = $gameMap.regionId(x, y);
      return blockingSwitch && BlockedRegionIds.includes(regionId);
    },
    getBlockedRegions: function() {
      return BlockedRegionIds;
    },
    getSwitchId: function() {
      return RegionBlockSwitchId;
    },
    isSwitchOn: function() {
      return RegionBlockSwitchId && $gameSwitches.value(RegionBlockSwitchId);
    }
  };

  log('Plugin initialization complete');
})();
